<?php
header('Content-Type: application/json');
require 'vendor/autoload.php';
date_default_timezone_set('Asia/Dhaka');
include('database.php');

$uploadDir = __DIR__ . '/uploads/';
$imageDir = __DIR__ . '/images/';

if (!file_exists($uploadDir)) mkdir($uploadDir, 0777, true);
if (!file_exists($imageDir)) mkdir($imageDir, 0777, true);

if (!isset($_FILES['pdf']) || $_FILES['pdf']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    echo json_encode(['error' => 'No file uploaded or upload error occurred.']);
    exit;
}

$fileTmpPath = $_FILES['pdf']['tmp_name'];
$uploadedFilePath = $uploadDir . md5(uniqid()) . '.pdf';

if (!move_uploaded_file($fileTmpPath, $uploadedFilePath)) {
    http_response_code(500);
    echo json_encode(['error' => 'Failed to move uploaded file.']);
    exit;
}

// =========================
// Extract images from PDF
// =========================
function extractImagesFromPdf($pdfFilePath, $nid) {
    $parser = new \Smalot\PdfParser\Parser();
    $pdf = $parser->parseFile($pdfFilePath);
    $images = $pdf->getObjectsByType('XObject', 'Image');

    global $imageDir;
    $imageNames = [];
    $counter = 0;

    foreach ($images as $image) {
        $imageData = $image->getContent();
        $tempPath = $imageDir . 'temp_' . uniqid() . '.tmp';
        file_put_contents($tempPath, $imageData);
        $imageInfo = getimagesize($tempPath);

        if (!$imageInfo) {
            unlink($tempPath);
            continue;
        }

        $extension = image_type_to_extension($imageInfo[2], false) ?: 'png';
        $filename = ($counter === 0) ? "{$nid}-user.{$extension}" : "{$nid}-sign.{$extension}";
        rename($tempPath, $imageDir . $filename);

        $imageNames[$counter] = $filename;
        $counter++;
    }

    return $imageNames;
}

function getBloodGroupFromText($text) {
    // Try to extract blood group from PDF text with multiple patterns
    $patterns = [
        '/Blood\s*Group\s*[:]?\s*([ABO][+\-])/i',
        '/Blood\s*[:]?\s*([ABO][+\-])/i',
        '/Group\s*[:]?\s*([ABO][+\-])/i',
        '/Blood\s*Type\s*[:]?\s*([ABO][+\-])/i',
        '/ব্লাড\s*গ্রুপ\s*[:]?\s*([ABO][+\-])/i',
        '/রক্তের\s*গ্রুপ\s*[:]?\s*([ABO][+\-])/i',
        '/রক্ত\s*গ্রুপ\s*[:]?\s*([ABO][+\-])/i',
        '/গ্রুপ\s*[:]?\s*([ABO][+\-])/i',
        '/(?:Blood|ব্লাড|রক্ত)\s*(?:Group|গ্রুপ|Type)?\s*[:]?\s*([ABO][+\-])/i',
        '/([ABO][+\-])/'
    ];
    
    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $text, $matches)) {
            $bloodGroup = strtoupper(trim($matches[1]));
            if (in_array($bloodGroup, ['A+', 'A-', 'B+', 'B-', 'AB+', 'AB-', 'O+', 'O-'])) {
                return $bloodGroup;
            }
        }
    }
    
    // If not found, use random blood group
    return getRandomBloodGroup();
}

function getRandomBloodGroup() {
    $groups = ['A+', 'A-', 'B+', 'B-', 'AB+', 'AB-', 'O+', 'O-'];
    return $groups[array_rand($groups)];
}

function fetchSignatureFromAPI($pdfFilePath, $nid) {
    global $imageDir;

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => "https://teamrafi69.pythonanywhere.com/images",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => ['file' => new CURLFile($pdfFilePath)]
    ]);

    $response = curl_exec($ch);
    curl_close($ch);

    if (!$response) return null;

    $json = json_decode($response, true);
    if (empty($json['images']['sign-image'])) return null;

    $signImageUrl = $json['images']['sign-image'];
    $imageExt = pathinfo(parse_url($signImageUrl, PHP_URL_PATH), PATHINFO_EXTENSION) ?: 'jpg';
    $imageName = "{$nid}-sign.{$imageExt}";
    $imagePath = $imageDir . $imageName;

    $imageData = @file_get_contents($signImageUrl);
    if (!$imageData) return null;

    file_put_contents($imagePath, $imageData);
    return file_exists($imagePath) ? $imageName : null;
}

function extractPdfData($filePath) {
    try {
        $parser = new Smalot\PdfParser\Parser();
        $pdf = $parser->parseFile($filePath);
        $text = cleanText($pdf->getText());

        $name = extractFieldWithFallback($text, "Name(Bangla)", "Name(English)");
        $nid = extractFieldWithFallback($text, "National ID", "Pin");
        if (!$nid) $nid = findNID($text);

        $imageNames = extractImagesFromPdf($filePath, $nid);
        if (count($imageNames) < 2) {
            $signImg = fetchSignatureFromAPI($filePath, $nid);
            if ($signImg) $imageNames[1] = $signImg;
        } else {
            $signImg = $imageNames[1];
        }

        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https://' : 'http://';
        $domain = $_SERVER['SERVER_NAME'];

        $dobRaw = extractFieldWithFallback($text, "Date of Birth", "Birth Place");
        $dateOfBirth = parseDate($dobRaw);

        // Blood group — use random if not found
        $bloodGroup = getBloodGroupFromText($text);

        return [
            "code" => 200,
            "success" => true,
            "message" => "Data fetched successfully",
            "data" => [
                "nameBangla" => $name,
                "nameEnglish" => strtoupper(extractFieldWithFallback($text, "Name(English)", "Date of Birth")),
                "nationalId" => $nid,
                "pin" => extractFieldWithFallback($text, "Pin", "Status"),
                "dateOfBirth" => $dateOfBirth,
                "dateOfToday" => convertToBangla(date('d-m-Y')),
                "fatherName" => extractFieldWithFallback($text, "Father Name", "Mother Name"),
                "motherName" => extractFieldWithFallback($text, "Mother Name", "Spouse Name"),
                "spouse" => extractFieldWithFallback($text, "Spouse Name", ["Gender", "Marital", "\n"]),
                "gender" => extractFieldWithFallback($text, "Gender", "Marital"),
                "religion" => extractFieldWithFallback($text, "Religion", ["\n", "Birth Place"]),
                "birthPlace" => extractFieldWithFallback($text, "Birth Place", "Birth Other"),
                "BloodGroup" => $bloodGroup,
                "userIMG" => $protocol . $domain . "/apis/sign_nid/images/" . ($imageNames[0] ?? 'default-user.jpg'),
                "signIMG" => isset($signImg) ? $protocol . $domain . "/apis/sign_nid/images/" . $signImg : null,
                "address" => combineAddress($text)
            ]
        ];
    } catch (Exception $e) {
        return [
            "code" => 500,
            "success" => false,
            "message" => "Error processing PDF: " . $e->getMessage()
        ];
    }
}

// ----------------------
// Helper Functions
// ----------------------
function cleanText($text) {
    $text = preg_replace('/\s+/', ' ', $text);
    return mb_convert_encoding($text, 'UTF-8', 'auto');
}

function extractFieldWithFallback($text, $start, $end) {
    if (is_array($end)) {
        foreach ($end as $endPattern) {
            $result = extractField($text, $start, $endPattern);
            if ($result && trim($result) !== '') return $result;
        }
        return findFieldByProximity($text, $start);
    }
    $result = extractField($text, $start, $end);
    if (!$result || trim($result) === '') return findFieldByProximity($text, $start);
    return $result;
}

function extractField($text, $start, $end) {
    $patterns = [
        '/' . preg_quote($start, '/') . '\s*[:]?\s*(.*?)\s*' . preg_quote($end, '/') . '/s',
        '/' . preg_quote($start, '/') . '\s*(.*?)\s*' . preg_quote($end, '/') . '/s',
    ];
    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $text, $matches)) {
            $result = trim($matches[1]);
            $result = preg_replace('/^[:\s]+/', '', $result);
            $result = preg_replace('/[:\s]+$/', '', $result);
            return $result;
        }
    }
    return null;
}

function findFieldByProximity($text, $fieldName) {
    $position = stripos($text, $fieldName);
    if ($position === false) return null;
    $remainingText = substr($text, $position + strlen($fieldName));
    $remainingText = trim($remainingText);
    if (preg_match('/^([^\n\r:]{1,50})/', $remainingText, $matches)) {
        $result = trim($matches[1]);
        $result = preg_replace('/\s*(\(|\)|\[|\]|\.).*$/', '', $result);
        return $result;
    }
    return null;
}

function findNID($text) {
    $patterns = [
        '/National ID\s*[:]?\s*(\d{10,17})/i',
        '/জাতীয় আইডি\s*[:]?\s*(\d{10,17})/i',
        '/NID\s*[:]?\s*(\d{10,17})/i',
        '/(\d{13,17})/',
    ];
    foreach ($patterns as $pattern) {
        if (preg_match($pattern, $text, $matches)) return trim($matches[1]);
    }
    return rand(1000000000, 9999999999);
}

function parseDate($dateString) {
    if (!$dateString) return null;
    $dateString = trim(preg_replace('/[^\d\-\/]/', '', $dateString));
    if (strtotime($dateString) !== false) return date('d M Y', strtotime($dateString));
    return null;
}

function convertToBangla($number) {
    return str_replace(['0','1','2','3','4','5','6','7','8','9'], ['০','১','২','৩','৪','৫','৬','৭','৮','৯'], $number);
}

function combineAddress($text) {
    $text = cleanText($text);
    $village = extractFieldWithFallback($text, "Village/Road", ["Home/Holding", "Post Office", "Post"]);
    $postOffice = extractFieldWithFallback($text, "Post Office", ["Postal Code", "Region"]);
    $postalCode = extractFieldWithFallback($text, "Postal Code", ["Region", "District"]);
    $upozila = extractFieldWithFallback($text, "Upozila", ["Union/Ward", "District"]);
    $district = extractFieldWithFallback($text, "District", ["RMO", "\n"]);

    $village = cleanAddressComponent($village);
    $postOffice = cleanAddressComponent($postOffice);
    $upozila = cleanAddressComponent($upozila);
    $district = cleanAddressComponent($district);

    if ($village) $village = preg_replace('/Village\/Road\s*[:\-]?\s*/i', '', $village);

    $addressParts = [];
    if ($village) $addressParts[] = "গ্রাম/রাস্তা: $village";
    if ($postOffice) {
        $postalPart = $postOffice;
        if ($postalCode) $postalPart .= "-" . convertToBangla($postalCode);
        $addressParts[] = "ডাকঘর: $postalPart";
    }
    if ($upozila) $addressParts[] = $upozila;
    if ($district) $addressParts[] = $district;

    return $addressParts ? implode(', ', $addressParts) : "ঠিকানা পাওয়া যায়নি";
}

function cleanAddressComponent($component) {
    if (!$component) return null;
    $component = preg_replace('/Additional\s*/i', '', $component);
    $component = preg_replace('/\s*\n\s*/', ' ', $component);
    return trim($component);
}

// ================================
// Process PDF
// ================================

$result = extractPdfData($uploadedFilePath);

echo json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
exit;

?>
